<?php
require_once 'auth.php';
requireAdminAuth();

require_once '../config/database.php';
require_once 'db.php';

$order_id = (int)($_GET['id'] ?? 0);

if ($order_id <= 0) {
    header('Location: orders.php');
    exit();
}

// Get order details
$order_stmt = $conn->prepare("
    SELECT 
        o.*,
        u.full_name as customer_name,
        u.email as customer_email,
        u.phone as customer_phone
    FROM orders o 
    LEFT JOIN users u ON o.user_id = u.id 
    WHERE o.id = ?
");
$order_stmt->bind_param("i", $order_id);
$order_stmt->execute();
$order_result = $order_stmt->get_result();

if ($order_result->num_rows === 0) {
    header('Location: orders.php');
    exit();
}

$order = $order_result->fetch_assoc();

// Get order items
$items_stmt = $conn->prepare("
    SELECT 
        oi.*,
        p.name as product_name,
        p.image_url as product_image
    FROM order_items oi
    LEFT JOIN products p ON oi.product_id = p.id
    WHERE oi.order_id = ?
");
$items_stmt->bind_param("i", $order_id);
$items_stmt->execute();
$items_result = $items_stmt->get_result();

$admin_info = getAdminInfo();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order #<?php echo $order['id']; ?> - SavingVest Mart Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f0f9ff',
                            100: '#e0f2fe',
                            200: '#bae6fd',
                            300: '#7dd3fc',
                            400: '#38bdf8',
                            500: '#0891b2',
                            600: '#0e7490',
                            700: '#0f766e',
                            800: '#115e59',
                            900: '#134e4a'
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-100">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <div class="flex items-center space-x-3">
                    <div class="w-8 h-8 bg-primary-500 rounded-lg flex items-center justify-center">
                        <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path>
                        </svg>
                    </div>
                    <span class="text-xl font-bold text-gray-900">SavingVest Admin</span>
                </div>

                <div class="hidden md:flex items-center space-x-8">
                    <a href="index.php" class="text-gray-600 hover:text-gray-900">Dashboard</a>
                    <a href="orders.php" class="text-primary-600 font-medium">Orders</a>
                    <a href="products.php" class="text-gray-600 hover:text-gray-900">Products</a>
                    <a href="categories.php" class="text-gray-600 hover:text-gray-900">Categories</a>
                    <a href="users.php" class="text-gray-600 hover:text-gray-900">Users</a>
                </div>

                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600">Welcome, <?php echo htmlspecialchars($admin_info['name']); ?></span>
                    <a href="logout.php" class="bg-red-500 text-white px-4 py-2 rounded-lg hover:bg-red-600 text-sm">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Page Header -->
        <div class="mb-8">
            <div class="flex items-center space-x-4">
                <a href="orders.php" class="text-primary-600 hover:text-primary-700">
                    ← Back to Orders
                </a>
            </div>
            <h1 class="text-3xl font-bold text-gray-900 mt-4">Order #<?php echo $order['id']; ?></h1>
            <p class="text-gray-600 mt-2">Order placed on <?php echo date('F j, Y \a\t g:i A', strtotime($order['created_at'])); ?></p>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Order Items -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-xl shadow-sm">
                    <div class="p-6 border-b border-gray-200">
                        <h2 class="text-lg font-semibold text-gray-900">Order Items</h2>
                    </div>
                    <div class="p-6">
                        <div class="space-y-4">
                            <?php while ($item = $items_result->fetch_assoc()): ?>
                                <div class="flex items-center space-x-4 p-4 bg-gray-50 rounded-lg">
                                    <img src="<?php echo htmlspecialchars($item['product_image'] ?? '/placeholder.svg?height=60&width=60&text=Product'); ?>" 
                                         alt="<?php echo htmlspecialchars($item['product_name']); ?>"
                                         class="w-16 h-16 object-cover rounded-lg">
                                    <div class="flex-1">
                                        <h3 class="font-medium text-gray-900"><?php echo htmlspecialchars($item['product_name']); ?></h3>
                                        <p class="text-sm text-gray-600">Quantity: <?php echo $item['quantity']; ?></p>
                                        <p class="text-sm text-gray-600">Price: ₦<?php echo number_format($item['price']); ?> each</p>
                                    </div>
                                    <div class="text-right">
                                        <p class="font-medium text-gray-900">₦<?php echo number_format($item['price'] * $item['quantity']); ?></p>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Order Summary & Customer Info -->
            <div class="space-y-6">
                <!-- Order Summary -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h2 class="text-lg font-semibold text-gray-900 mb-4">Order Summary</h2>
                    <div class="space-y-3">
                        <div class="flex justify-between">
                            <span class="text-gray-600">Subtotal:</span>
                            <span class="font-medium">₦<?php echo number_format($order['subtotal']); ?></span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-600">Shipping:</span>
                            <span class="font-medium">₦<?php echo number_format($order['shipping_cost']); ?></span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-600">Tax:</span>
                            <span class="font-medium">₦<?php echo number_format($order['tax_amount']); ?></span>
                        </div>
                        <div class="border-t pt-3">
                            <div class="flex justify-between">
                                <span class="text-lg font-semibold">Total:</span>
                                <span class="text-lg font-semibold">₦<?php echo number_format($order['total_amount']); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mt-6">
                        <span class="inline-flex px-3 py-1 text-sm font-medium rounded-full 
                            <?php 
                            switch($order['status']) {
                                case 'pending': echo 'bg-yellow-100 text-yellow-800'; break;
                                case 'processing': echo 'bg-blue-100 text-blue-800'; break;
                                case 'shipped': echo 'bg-purple-100 text-purple-800'; break;
                                case 'delivered': echo 'bg-green-100 text-green-800'; break;
                                case 'cancelled': echo 'bg-red-100 text-red-800'; break;
                                default: echo 'bg-gray-100 text-gray-800';
                            }
                            ?>">
                            <?php echo ucfirst($order['status']); ?>
                        </span>
                    </div>
                </div>

                <!-- Customer Information -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h2 class="text-lg font-semibold text-gray-900 mb-4">Customer Information</h2>
                    <div class="space-y-3">
                        <div>
                            <span class="text-sm text-gray-600">Name:</span>
                            <p class="font-medium"><?php echo htmlspecialchars($order['customer_name'] ?? 'Guest'); ?></p>
                        </div>
                        <div>
                            <span class="text-sm text-gray-600">Email:</span>
                            <p class="font-medium"><?php echo htmlspecialchars($order['customer_email'] ?? 'N/A'); ?></p>
                        </div>
                        <div>
                            <span class="text-sm text-gray-600">Phone:</span>
                            <p class="font-medium"><?php echo htmlspecialchars($order['customer_phone'] ?? 'N/A'); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Shipping Address -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h2 class="text-lg font-semibold text-gray-900 mb-4">Shipping Address</h2>
                    <div class="text-gray-600">
                        <?php echo nl2br(htmlspecialchars($order['shipping_address'])); ?>
                    </div>
                </div>

                <!-- Payment Information -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h2 class="text-lg font-semibold text-gray-900 mb-4">Payment Information</h2>
                    <div class="space-y-2">
                        <div>
                            <span class="text-sm text-gray-600">Method:</span>
                            <p class="font-medium"><?php echo ucfirst($order['payment_method']); ?></p>
                        </div>
                        <div>
                            <span class="text-sm text-gray-600">Status:</span>
                            <p class="font-medium"><?php echo ucfirst($order['payment_status']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
