<?php
require_once 'auth.php';
requireAdminAuth();

require_once '../config/database.php';
require_once 'db.php';

$product_id = (int)($_GET['id'] ?? 0);
$is_edit = $product_id > 0;
$product = null;

// Get product data for editing
if ($is_edit) {
    $product_stmt = $conn->prepare("SELECT * FROM products WHERE id = ?");
    $product_stmt->bind_param("i", $product_id);
    $product_stmt->execute();
    $product_result = $product_stmt->get_result();
    
    if ($product_result->num_rows === 0) {
        header('Location: products.php');
        exit();
    }
    
    $product = $product_result->fetch_assoc();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = (float)($_POST['price'] ?? 0);
    $category_id = (int)($_POST['category_id'] ?? 0);
    $sku = trim($_POST['sku'] ?? '');
    $quantity = (int)($_POST['quantity'] ?? 0);
    $status = $_POST['status'] ?? 'active';
    $specifications = trim($_POST['specifications'] ?? '');
    $features = trim($_POST['features'] ?? '');
    $brand = trim($_POST['brand'] ?? '');
    $weight = trim($_POST['weight'] ?? '');
    $dimensions = trim($_POST['dimensions'] ?? '');
    $image_url = trim($_POST['image_url'] ?? '');
    
    $errors = [];
    
    // Validation
    if (empty($name)) $errors[] = "Product name is required.";
    if (empty($description)) $errors[] = "Description is required.";
    if ($price <= 0) $errors[] = "Price must be greater than 0.";
    if ($category_id <= 0) $errors[] = "Please select a category.";
    if ($quantity < 0) $errors[] = "Stock quantity cannot be negative.";
    
    // Check SKU uniqueness
    if (!empty($sku)) {
        $sku_check_query = "SELECT id FROM products WHERE sku = ?";
        if ($is_edit) {
            $sku_check_query .= " AND id != ?";
        }
        
        $sku_stmt = $conn->prepare($sku_check_query);
        if ($is_edit) {
            $sku_stmt->bind_param("si", $sku, $product_id);
        } else {
            $sku_stmt->bind_param("s", $sku);
        }
        $sku_stmt->execute();
        
        if ($sku_stmt->get_result()->num_rows > 0) {
            $errors[] = "SKU already exists. Please use a unique SKU.";
        }
    }
    
    if (empty($errors)) {
        if ($is_edit) {
            // Update product
            $update_stmt = $conn->prepare("
                UPDATE products SET 
                    name = ?, description = ?, price = ?, category_id = ?, sku = ?, 
                    quantity = ?, status = ?, specifications = ?, features = ?, 
                    brand = ?, weight = ?, dimensions = ?, image_url = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $update_stmt->bind_param("ssdisssssssssi", 
                $name, $description, $price, $category_id, $sku, 
                $quantity, $status, $specifications, $features, 
                $brand, $weight, $dimensions, $image_url, $product_id
            );
            
            if ($update_stmt->execute()) {
                header('Location: products.php?updated=1');
                exit();
            } else {
                $errors[] = "Failed to update product.";
            }
        } else {
            // Create new product
            $insert_stmt = $conn->prepare("
                INSERT INTO products (
                    name, description, price, category_id, sku, quantity, 
                    status, specifications, features, brand, weight, dimensions, 
                    image_url, created_at, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
            ");
            $insert_stmt->bind_param("ssdississssss", 
                $name, $description, $price, $category_id, $sku, $quantity, 
                $status, $specifications, $features, $brand, $weight, $dimensions, $image_url
            );
            
            if ($insert_stmt->execute()) {
                header('Location: products.php?created=1');
                exit();
            } else {
                $errors[] = "Failed to create product.";
            }
        }
    }
}

// Get categories
$categories_result = $conn->query("SELECT id, name FROM categories ORDER BY name");

$admin_info = getAdminInfo();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $is_edit ? 'Edit' : 'Add'; ?> Product - SavingVest Mart Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f0f9ff',
                            100: '#e0f2fe',
                            200: '#bae6fd',
                            300: '#7dd3fc',
                            400: '#38bdf8',
                            500: '#0891b2',
                            600: '#0e7490',
                            700: '#0f766e',
                            800: '#115e59',
                            900: '#134e4a'
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-100">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <div class="flex items-center space-x-3">
                    <div class="w-8 h-8 bg-primary-500 rounded-lg flex items-center justify-center">
                        <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path>
                        </svg>
                    </div>
                    <span class="text-xl font-bold text-gray-900">SavingVest Admin</span>
                </div>

                <div class="hidden md:flex items-center space-x-8">
                    <a href="index.php" class="text-gray-600 hover:text-gray-900">Dashboard</a>
                    <a href="orders.php" class="text-gray-600 hover:text-gray-900">Orders</a>
                    <a href="products.php" class="text-primary-600 font-medium">Products</a>
                    <a href="categories.php" class="text-gray-600 hover:text-gray-900">Categories</a>
                    <a href="users.php" class="text-gray-600 hover:text-gray-900">Users</a>
                </div>

                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600">Welcome, <?php echo htmlspecialchars($admin_info['name']); ?></span>
                    <a href="logout.php" class="bg-red-500 text-white px-4 py-2 rounded-lg hover:bg-red-600 text-sm">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Page Header -->
        <div class="mb-8">
            <div class="flex items-center space-x-4">
                <a href="products.php" class="text-primary-600 hover:text-primary-700">
                    ← Back to Products
                </a>
            </div>
            <h1 class="text-3xl font-bold text-gray-900 mt-4">
                <?php echo $is_edit ? 'Edit Product' : 'Add New Product'; ?>
            </h1>
        </div>

        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
                <ul class="list-disc list-inside">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <!-- Product Form -->
        <form method="POST" class="bg-white rounded-xl shadow-sm">
            <div class="p-6 space-y-6">
                <!-- Basic Information -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Product Name *</label>
                        <input type="text" name="name" value="<?php echo htmlspecialchars($product['name'] ?? $_POST['name'] ?? ''); ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent" 
                               required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">SKU</label>
                        <input type="text" name="sku" value="<?php echo htmlspecialchars($product['sku'] ?? $_POST['sku'] ?? ''); ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                    </div>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Description *</label>
                    <textarea name="description" rows="4" 
                              class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent" 
                              required><?php echo htmlspecialchars($product['description'] ?? $_POST['description'] ?? ''); ?></textarea>
                </div>

                <!-- Pricing and Inventory -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Price (₦) *</label>
                        <input type="number" name="price" step="0.01" min="0" 
                               value="<?php echo $product['price'] ?? $_POST['price'] ?? ''; ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent" 
                               required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Stock Quantity *</label>
                        <input type="number" name="quantity" min="0" 
                               value="<?php echo $product['quantity'] ?? $_POST['quantity'] ?? ''; ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent" 
                               required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                        <select name="status" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                            <option value="active" <?php echo ($product['status'] ?? $_POST['status'] ?? 'active') === 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo ($product['status'] ?? $_POST['status'] ?? '') === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>
                </div>

                <!-- Category and Brand -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Category *</label>
                        <select name="category_id" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent" required>
                            <option value="">Select Category</option>
                            <?php while ($category = $categories_result->fetch_assoc()): ?>
                                <option value="<?php echo $category['id']; ?>" 
                                        <?php echo ($product['category_id'] ?? $_POST['category_id'] ?? '') == $category['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($category['name']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Brand</label>
                        <input type="text" name="brand" value="<?php echo htmlspecialchars($product['brand'] ?? $_POST['brand'] ?? ''); ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                    </div>
                </div>

                <!-- Image URL -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Image URL</label>
                    <input type="url" name="image_url" value="<?php echo htmlspecialchars($product['image_url'] ?? $_POST['image_url'] ?? ''); ?>" 
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                           placeholder="https://example.com/image.jpg">
                </div>

                <!-- Physical Properties -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Weight</label>
                        <input type="text" name="weight" value="<?php echo htmlspecialchars($product['weight'] ?? $_POST['weight'] ?? ''); ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                               placeholder="e.g., 2.5kg">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Dimensions</label>
                        <input type="text" name="dimensions" value="<?php echo htmlspecialchars($product['dimensions'] ?? $_POST['dimensions'] ?? ''); ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                               placeholder="e.g., 30cm x 20cm x 15cm">
                    </div>
                </div>

                <!-- Specifications -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Specifications</label>
                    <textarea name="specifications" rows="4" 
                              class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                              placeholder="Technical specifications, one per line"><?php echo htmlspecialchars($product['specifications'] ?? $_POST['specifications'] ?? ''); ?></textarea>
                </div>

                <!-- Features -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Features</label>
                    <textarea name="features" rows="4" 
                              class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                              placeholder="Key features, one per line"><?php echo htmlspecialchars($product['features'] ?? $_POST['features'] ?? ''); ?></textarea>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="px-6 py-4 bg-gray-50 border-t border-gray-200 rounded-b-xl">
                <div class="flex justify-end space-x-4">
                    <a href="products.php" class="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                        Cancel
                    </a>
                    <button type="submit" class="px-6 py-2 bg-primary-500 text-white rounded-lg hover:bg-primary-600">
                        <?php echo $is_edit ? 'Update Product' : 'Create Product'; ?>
                    </button>
                </div>
            </div>
        </form>
    </div>
</body>
</html>
