<?php
$page_title = "My Orders";
$page_description = "View and track all your orders.";

require_once 'config/database.php';
include 'includes/header.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?redirect=" . urlencode($_SERVER['REQUEST_URI']));
    exit;
}

$conn = getDBConnection();
$user_id = $_SESSION['user_id'];

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Get total orders count
$count_sql = "SELECT COUNT(*) as total FROM orders WHERE user_id = ?";
$count_stmt = $conn->prepare($count_sql);
$count_stmt->bind_param("i", $user_id);
$count_stmt->execute();
$total_orders = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_orders / $per_page);

// Get orders
$orders_sql = "SELECT * FROM orders WHERE user_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?";
$orders_stmt = $conn->prepare($orders_sql);
$orders_stmt->bind_param("iii", $user_id, $per_page, $offset);
$orders_stmt->execute();
$orders = $orders_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$conn->close();
?>

<!-- Orders Header -->
<section class="bg-gray-100 py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="flex items-center space-x-4">
            <a href="dashboard.php" class="text-gray-600 hover:text-primary-600">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
            </a>
            <div>
                <h1 class="text-3xl font-bold text-gray-900">My Orders</h1>
                <p class="text-gray-600 mt-2"><?php echo $total_orders; ?> order<?php echo $total_orders !== 1 ? 's' : ''; ?> found</p>
            </div>
        </div>
    </div>
</section>

<!-- Orders List -->
<section class="py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <?php if (empty($orders)): ?>
            <!-- No Orders -->
            <div class="bg-white rounded-lg shadow-sm p-12 text-center">
                <svg class="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"></path>
                </svg>
                <h2 class="text-2xl font-bold text-gray-900 mb-4">No orders yet</h2>
                <p class="text-gray-600 mb-8">Start shopping to see your orders here.</p>
                <a href="products.php" class="bg-primary-500 text-white px-8 py-3 rounded-lg font-semibold hover:bg-primary-600">
                    Browse Products
                </a>
            </div>
        <?php else: ?>
            <!-- Orders Grid -->
            <div class="space-y-6">
                <?php foreach ($orders as $order): ?>
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Order #<?php echo htmlspecialchars($order['order_number']); ?></h3>
                                <p class="text-sm text-gray-600">Placed on <?php echo date('F j, Y', strtotime($order['created_at'])); ?></p>
                            </div>
                            <div class="text-right">
                                <p class="text-lg font-bold text-gray-900">$<?php echo number_format($order['total_amount'], 2); ?></p>
                                <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium
                                    <?php 
                                    switch($order['status']) {
                                        case 'pending': echo 'bg-yellow-100 text-yellow-800'; break;
                                        case 'processing': echo 'bg-blue-100 text-blue-800'; break;
                                        case 'shipped': echo 'bg-purple-100 text-purple-800'; break;
                                        case 'delivered': echo 'bg-green-100 text-green-800'; break;
                                        case 'cancelled': echo 'bg-red-100 text-red-800'; break;
                                        default: echo 'bg-gray-100 text-gray-800';
                                    }
                                    ?>">
                                    <?php echo ucfirst($order['status']); ?>
                                </span>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                            <div>
                                <span class="text-gray-600">Payment:</span>
                                <span class="ml-2 font-medium text-gray-900"><?php echo ucfirst(str_replace('_', ' ', $order['payment_method'])); ?></span>
                            </div>
                            <div>
                                <span class="text-gray-600">Payment Status:</span>
                                <span class="ml-2 font-medium <?php echo $order['payment_status'] === 'paid' ? 'text-green-600' : 'text-yellow-600'; ?>">
                                    <?php echo ucfirst($order['payment_status']); ?>
                                </span>
                            </div>
                            <div>
                                <span class="text-gray-600">Last Updated:</span>
                                <span class="ml-2 font-medium text-gray-900"><?php echo date('M j, Y', strtotime($order['updated_at'])); ?></span>
                            </div>
                        </div>
                        
                        <div class="mt-4 flex items-center justify-between">
                            <a href="order-details.php?id=<?php echo $order['id']; ?>" 
                               class="text-primary-600 hover:text-primary-800 font-medium">
                                View Details →
                            </a>
                            
                            <?php if ($order['status'] === 'pending'): ?>
                                <button onclick="cancelOrder(<?php echo $order['id']; ?>)" 
                                        class="text-red-600 hover:text-red-800 text-sm font-medium">
                                    Cancel Order
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <div class="mt-8 flex justify-center">
                <nav class="flex items-center space-x-2">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?php echo $page - 1; ?>" 
                           class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                            Previous
                        </a>
                    <?php endif; ?>
                    
                    <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                        <?php if ($i == $page): ?>
                            <span class="px-3 py-2 bg-primary-500 text-white rounded-md"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?page=<?php echo $i; ?>" 
                               class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50"><?php echo $i; ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    
                    <?php if ($page < $total_pages): ?>
                        <a href="?page=<?php echo $page + 1; ?>" 
                           class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                            Next
                        </a>
                    <?php endif; ?>
                </nav>
            </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</section>

<script>
function cancelOrder(orderId) {
    if (confirm('Are you sure you want to cancel this order?')) {
        // In a real implementation, this would make an API call to cancel the order
        showNotification('Order cancellation requested. You will receive a confirmation email.', 'info');
    }
}
</script>

<?php include 'includes/footer.php'; ?>
