-- Complete Database Setup for SavingVest Mart E-commerce Platform
-- This script will drop all existing tables and recreate the entire database structure with sample data

-- Drop existing tables (in correct order to handle foreign key constraints)
DROP TABLE IF EXISTS order_items;
DROP TABLE IF EXISTS orders;
DROP TABLE IF EXISTS cart_items;
DROP TABLE IF EXISTS reviews;
DROP TABLE IF EXISTS products;
DROP TABLE IF EXISTS categories;
DROP TABLE IF EXISTS users;

-- Create users table
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    address TEXT,
    city VARCHAR(50),
    state VARCHAR(50),
    postal_code VARCHAR(10),
    role ENUM('user', 'admin') DEFAULT 'user',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Create categories table
CREATE TABLE categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    slug VARCHAR(100) UNIQUE NOT NULL,
    description TEXT,
    image_url VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Create products table
CREATE TABLE products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    slug VARCHAR(255) UNIQUE NOT NULL,
    description TEXT,
    short_description TEXT,
    price DECIMAL(10, 2) NOT NULL,
    compare_price DECIMAL(10, 2),
    cost_price DECIMAL(10, 2),
    sku VARCHAR(100) UNIQUE,
    barcode VARCHAR(100),
    track_quantity BOOLEAN DEFAULT TRUE,
    quantity INT DEFAULT 0,
    allow_backorder BOOLEAN DEFAULT FALSE,
    weight DECIMAL(8, 2),
    length DECIMAL(8, 2),
    width DECIMAL(8, 2),
    height DECIMAL(8, 2),
    category_id INT,
    brand VARCHAR(100),
    model VARCHAR(100),
    color VARCHAR(50),
    size VARCHAR(50),
    material VARCHAR(100),
    warranty_period VARCHAR(50),
    image_url VARCHAR(255),
    gallery_images TEXT,
    features TEXT,
    specifications TEXT,
    is_featured BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    meta_title VARCHAR(255),
    meta_description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Create cart_items table
CREATE TABLE cart_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    product_id INT NOT NULL,
    quantity INT NOT NULL DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Create orders table
CREATE TABLE orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    order_number VARCHAR(50) UNIQUE NOT NULL,
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    total_amount DECIMAL(10, 2) NOT NULL,
    subtotal DECIMAL(10, 2) NOT NULL,
    tax_amount DECIMAL(10, 2) DEFAULT 0,
    shipping_amount DECIMAL(10, 2) DEFAULT 0,
    discount_amount DECIMAL(10, 2) DEFAULT 0,
    payment_method VARCHAR(50),
    payment_status ENUM('pending', 'paid', 'failed', 'refunded') DEFAULT 'pending',
    shipping_address TEXT,
    billing_address TEXT,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Create order_items table
CREATE TABLE order_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    product_id INT NOT NULL,
    quantity INT NOT NULL,
    price DECIMAL(10, 2) NOT NULL,
    total DECIMAL(10, 2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Create reviews table
CREATE TABLE reviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_id INT NOT NULL,
    user_id INT,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    title VARCHAR(255),
    comment TEXT,
    is_verified BOOLEAN DEFAULT FALSE,
    is_approved BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Insert sample categories
INSERT INTO categories (name, slug, description, image_url) VALUES
('Smartphones', 'smartphones', 'Latest smartphones and mobile devices', '/images/categories/smartphones.jpg'),
('Household Appliances', 'household-appliances', 'Essential home appliances for modern living', '/images/categories/home-appliances.jpg'),
('Kitchen Appliances', 'kitchen-appliances', 'Modern kitchen appliances and gadgets', '/images/categories/home-appliances.jpg'),
('Electronics', 'electronics', 'Consumer electronics and gadgets', '/images/categories/electronics.jpg');

-- Insert sample users (including admin)
INSERT INTO users (first_name, last_name, email, password, phone, address, city, state, role) VALUES
('Admin', 'User', 'admin@savingvest.com', '$2y$10$YourHashedPasswordHere', '+2348038928708', 'Suite 138/139 Atara Shopping Plaza, Kushimo Street', 'Lagos', 'Lagos State', 'admin'),
('John', 'Doe', 'john@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '+2348012345678', '123 Victoria Island', 'Lagos', 'Lagos State', 'user'),
('Jane', 'Smith', 'jane@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '+2348087654321', '456 Ikeja GRA', 'Lagos', 'Lagos State', 'user');

-- Insert sample products
INSERT INTO products (name, slug, description, short_description, price, compare_price, sku, quantity, category_id, brand, model, image_url, features, specifications, is_featured) VALUES
('iPhone 15 Pro Max', 'iphone-15-pro-max', 'The most advanced iPhone with titanium design and powerful A17 Pro chip.', 'Latest iPhone with titanium design', 1200000.00, 1400000.00, 'IP15PM-001', 25, 1, 'Apple', 'iPhone 15 Pro Max', '/images/products/iphone-15-pro-max.jpg', 'A17 Pro chip, Titanium design, Advanced camera system, Action Button', 'Display: 6.7-inch Super Retina XDR, Storage: 256GB, Camera: 48MP Main', TRUE),

('Samsung Galaxy S24 Ultra', 'samsung-galaxy-s24-ultra', 'Premium Android smartphone with S Pen and advanced AI features.', 'Premium Samsung flagship with S Pen', 1100000.00, 1300000.00, 'SGS24U-001', 20, 1, 'Samsung', 'Galaxy S24 Ultra', '/images/products/samsung-galaxy-s24-ultra.jpg', 'S Pen included, AI-powered features, 200MP camera, 5000mAh battery', 'Display: 6.8-inch Dynamic AMOLED, Storage: 256GB, RAM: 12GB', TRUE),

('Dyson V15 Detect', 'dyson-v15-detect', 'Advanced cordless vacuum with laser dust detection technology.', 'Cordless vacuum with laser detection', 480000.00, 550000.00, 'DV15D-001', 15, 2, 'Dyson', 'V15 Detect', '/images/products/dyson-v15-detect.jpg', 'Laser dust detection, LCD screen, Up to 60 minutes runtime', 'Power: 230W, Weight: 3.1kg, Bin capacity: 0.77L', TRUE),

('KitchenAid Stand Mixer', 'kitchenaid-stand-mixer', 'Professional-grade stand mixer for all your baking needs.', 'Professional stand mixer', 320000.00, 380000.00, 'KASM-001', 12, 3, 'KitchenAid', 'Artisan Series', '/images/products/kitchenaid-mixer.jpg', '10-speed control, Tilt-head design, Multiple attachments available', 'Capacity: 4.8L, Power: 325W, Material: Die-cast zinc', FALSE),

('Google Pixel 8 Pro', 'google-pixel-8-pro', 'AI-powered smartphone with advanced computational photography.', 'Google flagship with AI features', 950000.00, 1100000.00, 'GP8P-001', 18, 1, 'Google', 'Pixel 8 Pro', '/images/products/google-pixel-8-pro.jpg', 'Tensor G3 chip, Magic Eraser, Night Sight, 7 years of updates', 'Display: 6.7-inch LTPO OLED, Storage: 128GB, Camera: 50MP Triple', FALSE),

('Ninja Air Fryer', 'ninja-air-fryer', 'Large capacity air fryer for healthy cooking with multiple functions.', 'Multi-function air fryer', 180000.00, 220000.00, 'NAF-001', 30, 3, 'Ninja', 'Foodi MAX', '/images/products/ninja-air-fryer.jpg', '9-in-1 functionality, 9.5L capacity, Dual Zone technology', 'Capacity: 9.5L, Power: 2470W, Functions: Air Fry, Roast, Bake, Reheat', FALSE),

('iPad Pro 12.9-inch', 'ipad-pro-12-inch', 'Professional tablet with M2 chip and Liquid Retina XDR display.', 'Professional iPad with M2 chip', 880000.00, 1000000.00, 'IPP12-001', 22, 4, 'Apple', 'iPad Pro', '/images/products/ipad-pro-12-inch.jpg', 'M2 chip, Liquid Retina XDR display, Apple Pencil support, Magic Keyboard compatible', 'Display: 12.9-inch Liquid Retina XDR, Storage: 256GB, Chip: Apple M2', TRUE),

('iRobot Roomba j7+', 'irobot-roomba-j7', 'Smart robot vacuum with obstacle avoidance and self-emptying base.', 'Smart robot vacuum with auto-empty', 560000.00, 650000.00, 'IRJ7-001', 10, 2, 'iRobot', 'Roomba j7+', '/images/products/irobot-roomba-j7.jpg', 'PrecisionVision Navigation, Self-emptying base, Pet waste avoidance', 'Runtime: 75 minutes, Bin capacity: 400ml, WiFi enabled', FALSE),

('LG OLED C3 55-inch TV', 'lg-oled-c3-55-inch', 'Premium OLED TV with perfect blacks and vibrant colors.', '55-inch OLED smart TV', 1600000.00, 1800000.00, 'LGOC3-55', 8, 4, 'LG', 'OLED C3', '/images/products/lg-oled-tv.jpg', 'OLED evo technology, α9 Gen6 AI Processor, Dolby Vision IQ, webOS 23', 'Size: 55-inch, Resolution: 4K Ultra HD, HDR: Dolby Vision, Refresh: 120Hz', FALSE),

('Sony WH-1000XM5', 'sony-wh-1000xm5', 'Industry-leading noise canceling wireless headphones.', 'Premium noise-canceling headphones', 280000.00, 320000.00, 'SWH1000XM5', 35, 4, 'Sony', 'WH-1000XM5', '/images/products/sony-headphones.jpg', 'Industry-leading noise canceling, 30-hour battery, Multipoint connection', 'Driver: 30mm, Frequency: 4Hz-40kHz, Battery: 30 hours, Weight: 250g', FALSE),

('Instant Pot Duo 7-in-1', 'instant-pot-duo-7-in-1', 'Multi-functional electric pressure cooker for quick and easy meals.', '7-in-1 electric pressure cooker', 120000.00, 150000.00, 'IPD7-001', 25, 3, 'Instant Pot', 'Duo', '/images/products/instant-pot.jpg', '7 appliances in 1, 13 one-touch programs, Safe and convenient', 'Capacity: 6L, Functions: Pressure Cook, Slow Cook, Rice Cooker, Steamer', FALSE),

('MacBook Air M2', 'macbook-air-m2', 'Incredibly thin and light laptop with M2 chip and all-day battery.', 'Ultra-thin laptop with M2 chip', 1400000.00, 1600000.00, 'MBA-M2-001', 12, 4, 'Apple', 'MacBook Air', '/images/products/macbook-air.jpg', 'Apple M2 chip, 18-hour battery, Liquid Retina display, MagSafe charging', 'Display: 13.6-inch Liquid Retina, Storage: 256GB SSD, RAM: 8GB, Chip: Apple M2', TRUE);

-- Insert sample reviews
INSERT INTO reviews (product_id, user_id, rating, title, comment, is_verified) VALUES
(1, 2, 5, 'Amazing phone!', 'The iPhone 15 Pro Max is incredible. The camera quality is outstanding and the titanium build feels premium.', TRUE),
(1, 3, 4, 'Great but expensive', 'Love the features but the price is quite high. Overall satisfied with the purchase.', TRUE),
(2, 2, 5, 'Best Android phone', 'The S Pen functionality is amazing and the camera takes stunning photos.', TRUE),
(3, 3, 5, 'Game changer for cleaning', 'This vacuum has revolutionized my cleaning routine. The laser detection is so cool!', TRUE),
(7, 2, 5, 'Perfect for work', 'The M2 chip makes everything so smooth. Great for design work and video editing.', TRUE);

-- Insert sample orders
INSERT INTO orders (user_id, order_number, status, total_amount, subtotal, tax_amount, shipping_amount, payment_method, payment_status, shipping_address) VALUES
(2, 'ORD-2024-001', 'delivered', 1275000.00, 1200000.00, 75000.00, 0.00, 'paystack', 'paid', '123 Victoria Island, Lagos, Lagos State'),
(3, 'ORD-2024-002', 'processing', 1207500.00, 1100000.00, 82500.00, 25000.00, 'bank_transfer', 'paid', '456 Ikeja GRA, Lagos, Lagos State'),
(2, 'ORD-2024-003', 'shipped', 510000.00, 480000.00, 30000.00, 0.00, 'flutterwave', 'paid', '123 Victoria Island, Lagos, Lagos State');

-- Insert sample order items
INSERT INTO order_items (order_id, product_id, quantity, price, total) VALUES
(1, 1, 1, 1200000.00, 1200000.00),
(2, 2, 1, 1100000.00, 1100000.00),
(3, 3, 1, 480000.00, 480000.00);

-- Create indexes for better performance
CREATE INDEX idx_products_category ON products(category_id);
CREATE INDEX idx_products_featured ON products(is_featured);
CREATE INDEX idx_products_active ON products(is_active);
CREATE INDEX idx_orders_user ON orders(user_id);
CREATE INDEX idx_orders_status ON orders(status);
CREATE INDEX idx_order_items_order ON order_items(order_id);
CREATE INDEX idx_order_items_product ON order_items(product_id);
CREATE INDEX idx_reviews_product ON reviews(product_id);
CREATE INDEX idx_reviews_user ON reviews(user_id);
CREATE INDEX idx_cart_items_user ON cart_items(user_id);
CREATE INDEX idx_users_email ON users(email);
CREATE INDEX idx_users_role ON users(role);

-- Update admin password hash (you'll need to replace this with the actual hash)
-- Run this after generating the proper hash: UPDATE users SET password = 'your_generated_hash' WHERE email = 'admin@savingvest.com';
