<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Please log in']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['cart_item_id'])) {
    echo json_encode(['success' => false, 'message' => 'Missing cart item ID']);
    exit;
}

$user_id = $_SESSION['user_id'];
$cart_item_id = intval($input['cart_item_id']);

$conn = getDBConnection();

// Verify cart item belongs to user
$verify_sql = "SELECT id FROM cart_items WHERE id = ? AND user_id = ?";
$verify_stmt = $conn->prepare($verify_sql);
$verify_stmt->bind_param("ii", $cart_item_id, $user_id);
$verify_stmt->execute();

if (!$verify_stmt->get_result()->fetch_assoc()) {
    echo json_encode(['success' => false, 'message' => 'Cart item not found']);
    exit;
}

// Remove cart item
$delete_sql = "DELETE FROM cart_items WHERE id = ?";
$delete_stmt = $conn->prepare($delete_sql);
$delete_stmt->bind_param("i", $cart_item_id);

if ($delete_stmt->execute()) {
    // Get updated cart totals
    $totals_sql = "SELECT 
                    SUM(COALESCE(p.sale_price, p.price) * ci.quantity) as subtotal,
                    SUM(ci.quantity) as total_items
                   FROM cart_items ci
                   JOIN products p ON ci.product_id = p.id
                   WHERE ci.user_id = ?";
    $totals_stmt = $conn->prepare($totals_sql);
    $totals_stmt->bind_param("i", $user_id);
    $totals_stmt->execute();
    $totals = $totals_stmt->get_result()->fetch_assoc();
    
    $subtotal = floatval($totals['subtotal'] ?? 0);
    $shipping = $subtotal >= 99 ? 0 : 9.99;
    $tax = $subtotal * 0.08;
    $total = $subtotal + $shipping + $tax;
    
    echo json_encode([
        'success' => true,
        'cart_totals' => [
            'subtotal' => number_format($subtotal, 2),
            'shipping' => number_format($shipping, 2),
            'tax' => number_format($tax, 2),
            'total' => number_format($total, 2),
            'total_items' => intval($totals['total_items'] ?? 0)
        ]
    ]);
} else {
    echo json_encode(['success' => false, 'message' => 'Failed to remove item']);
}

$conn->close();
?>
