"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Separator } from "@/components/ui/separator"
import {
  ShoppingCart,
  Search,
  User,
  Menu,
  Plus,
  Minus,
  Trash2,
  ChevronRight,
  Truck,
  Shield,
  CreditCard,
  Lock,
} from "lucide-react"
import { useCart } from "@/lib/cart-context"

export default function CartPage() {
  const { state, dispatch } = useCart()
  const [promoCode, setPromoCode] = useState("")
  const [appliedPromo, setAppliedPromo] = useState<{ code: string; discount: number } | null>(null)

  const shipping = state.total > 50 ? 0 : 9.99
  const tax = state.total * 0.08 // 8% tax
  const discount = appliedPromo ? state.total * appliedPromo.discount : 0
  const finalTotal = state.total + shipping + tax - discount

  const updateQuantity = (id: number, quantity: number) => {
    dispatch({ type: "UPDATE_QUANTITY", payload: { id, quantity } })
  }

  const removeItem = (id: number) => {
    dispatch({ type: "REMOVE_ITEM", payload: id })
  }

  const applyPromoCode = () => {
    // Mock promo code logic
    if (promoCode.toLowerCase() === "save10") {
      setAppliedPromo({ code: promoCode, discount: 0.1 })
      setPromoCode("")
    }
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Navigation Header */}
      <header className="border-b bg-white/95 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-8">
              <h1 className="text-2xl font-bold text-emerald-600">SavingVest Mart</h1>
              <nav className="hidden md:flex items-center gap-6">
                <a href="/" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Home
                </a>
                <a href="/products" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Products
                </a>
                <a href="#" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Categories
                </a>
                <a href="#" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Deals
                </a>
              </nav>
            </div>

            <div className="flex items-center gap-4">
              <div className="hidden md:flex items-center gap-2 bg-gray-100 rounded-full px-4 py-2 w-80">
                <Search className="h-4 w-4 text-gray-500" />
                <input
                  type="text"
                  placeholder="Search products..."
                  className="bg-transparent border-none outline-none flex-1 text-sm"
                />
              </div>
              <Button variant="ghost" size="icon">
                <User className="h-5 w-5" />
              </Button>
              <Button variant="ghost" size="icon" className="relative">
                <ShoppingCart className="h-5 w-5" />
                <Badge className="absolute -top-2 -right-2 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs">
                  {state.itemCount}
                </Badge>
              </Button>
              <Button variant="ghost" size="icon" className="md:hidden">
                <Menu className="h-5 w-5" />
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Breadcrumb */}
      <div className="bg-gray-50 py-4">
        <div className="container mx-auto px-4">
          <div className="flex items-center gap-2 text-sm text-gray-600">
            <a href="/" className="hover:text-emerald-600">
              Home
            </a>
            <ChevronRight className="h-4 w-4" />
            <span className="text-gray-900">Shopping Cart</span>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-4 py-8">
        {state.items.length === 0 ? (
          <div className="text-center py-16">
            <ShoppingCart className="h-24 w-24 text-gray-300 mx-auto mb-4" />
            <h2 className="text-2xl font-bold text-gray-900 mb-2">Your cart is empty</h2>
            <p className="text-gray-600 mb-6">Add some products to get started!</p>
            <Button asChild className="bg-emerald-600 hover:bg-emerald-700">
              <a href="/products">Continue Shopping</a>
            </Button>
          </div>
        ) : (
          <div className="grid lg:grid-cols-3 gap-8">
            {/* Cart Items */}
            <div className="lg:col-span-2 space-y-6">
              <div className="flex items-center justify-between">
                <h1 className="text-3xl font-bold text-gray-900">Shopping Cart</h1>
                <span className="text-gray-600">{state.itemCount} items</span>
              </div>

              <div className="space-y-4">
                {state.items.map((item) => (
                  <Card key={`${item.id}-${item.storage}`} className="border-0 shadow-md">
                    <CardContent className="p-6">
                      <div className="flex gap-6">
                        <div className="w-24 h-24 bg-gray-100 rounded-lg overflow-hidden flex-shrink-0">
                          <img
                            src={item.image || "/placeholder.svg"}
                            alt={item.name}
                            className="w-full h-full object-cover"
                          />
                        </div>

                        <div className="flex-1 space-y-2">
                          <div className="flex items-start justify-between">
                            <div>
                              <h3 className="font-semibold text-lg text-gray-900">{item.name}</h3>
                              <div className="flex items-center gap-2 text-sm text-gray-600">
                                <Badge variant="outline">{item.brand}</Badge>
                                {item.storage && <span>• {item.storage}</span>}
                              </div>
                            </div>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => removeItem(item.id)}
                              className="text-red-600 hover:text-red-700 hover:bg-red-50"
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>

                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2">
                              <span className="text-xl font-bold text-gray-900">${item.price}</span>
                              {item.originalPrice && item.originalPrice > item.price && (
                                <span className="text-sm text-gray-500 line-through">${item.originalPrice}</span>
                              )}
                            </div>

                            <div className="flex items-center gap-3">
                              <div className="flex items-center border rounded-lg">
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => updateQuantity(item.id, item.quantity - 1)}
                                  disabled={item.quantity <= 1}
                                >
                                  <Minus className="h-4 w-4" />
                                </Button>
                                <span className="px-4 py-2 font-medium">{item.quantity}</span>
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => updateQuantity(item.id, item.quantity + 1)}
                                >
                                  <Plus className="h-4 w-4" />
                                </Button>
                              </div>
                              <span className="font-semibold text-lg w-20 text-right">
                                ${(item.price * item.quantity).toFixed(2)}
                              </span>
                            </div>
                          </div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>

              <div className="flex justify-between items-center pt-4">
                <Button variant="outline" asChild>
                  <a href="/products">Continue Shopping</a>
                </Button>
                <Button
                  variant="outline"
                  onClick={() => dispatch({ type: "CLEAR_CART" })}
                  className="text-red-600 hover:text-red-700 hover:bg-red-50"
                >
                  Clear Cart
                </Button>
              </div>
            </div>

            {/* Order Summary */}
            <div className="space-y-6">
              <Card className="border-0 shadow-md">
                <CardContent className="p-6 space-y-4">
                  <h2 className="text-xl font-semibold text-gray-900">Order Summary</h2>

                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span>Subtotal ({state.itemCount} items)</span>
                      <span>${state.total.toFixed(2)}</span>
                    </div>

                    <div className="flex justify-between">
                      <span>Shipping</span>
                      <span>{shipping === 0 ? "Free" : `$${shipping.toFixed(2)}`}</span>
                    </div>

                    <div className="flex justify-between">
                      <span>Tax</span>
                      <span>${tax.toFixed(2)}</span>
                    </div>

                    {appliedPromo && (
                      <div className="flex justify-between text-green-600">
                        <span>Discount ({appliedPromo.code})</span>
                        <span>-${discount.toFixed(2)}</span>
                      </div>
                    )}

                    <Separator />

                    <div className="flex justify-between text-lg font-semibold">
                      <span>Total</span>
                      <span>${finalTotal.toFixed(2)}</span>
                    </div>
                  </div>

                  {state.total < 50 && (
                    <div className="bg-blue-50 p-3 rounded-lg text-sm text-blue-700">
                      Add ${(50 - state.total).toFixed(2)} more for free shipping!
                    </div>
                  )}
                </CardContent>
              </Card>

              {/* Promo Code */}
              <Card className="border-0 shadow-md">
                <CardContent className="p-6 space-y-4">
                  <h3 className="font-semibold">Promo Code</h3>
                  <div className="flex gap-2">
                    <Input
                      placeholder="Enter promo code"
                      value={promoCode}
                      onChange={(e) => setPromoCode(e.target.value)}
                    />
                    <Button onClick={applyPromoCode} variant="outline">
                      Apply
                    </Button>
                  </div>
                  {appliedPromo && (
                    <div className="text-sm text-green-600">✓ Promo code "{appliedPromo.code}" applied!</div>
                  )}
                </CardContent>
              </Card>

              {/* Checkout Button */}
              <Button className="w-full bg-emerald-600 hover:bg-emerald-700 text-white py-3 text-lg">
                <Lock className="h-5 w-5 mr-2" />
                Proceed to Checkout
              </Button>

              {/* Security Features */}
              <div className="space-y-3 text-sm text-gray-600">
                <div className="flex items-center gap-2">
                  <Shield className="h-4 w-4 text-emerald-600" />
                  <span>Secure SSL encryption</span>
                </div>
                <div className="flex items-center gap-2">
                  <Truck className="h-4 w-4 text-emerald-600" />
                  <span>Free shipping on orders over $50</span>
                </div>
                <div className="flex items-center gap-2">
                  <CreditCard className="h-4 w-4 text-emerald-600" />
                  <span>Multiple payment options</span>
                </div>
              </div>

              {/* Payment Methods */}
              <Card className="border-0 shadow-md">
                <CardContent className="p-6">
                  <h3 className="font-semibold mb-3">We Accept</h3>
                  <div className="flex gap-2 flex-wrap">
                    <div className="bg-gray-100 px-3 py-2 rounded text-xs font-medium">VISA</div>
                    <div className="bg-gray-100 px-3 py-2 rounded text-xs font-medium">MASTERCARD</div>
                    <div className="bg-gray-100 px-3 py-2 rounded text-xs font-medium">AMEX</div>
                    <div className="bg-gray-100 px-3 py-2 rounded text-xs font-medium">PAYPAL</div>
                    <div className="bg-gray-100 px-3 py-2 rounded text-xs font-medium">APPLE PAY</div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        )}
      </div>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-16 mt-16">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-4 gap-8">
            <div className="space-y-4">
              <h3 className="text-2xl font-bold text-emerald-400">SavingVest Mart</h3>
              <p className="text-gray-400">Your trusted partner for premium products at unbeatable prices.</p>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Quick Links</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  About Us
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Contact
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Shipping Info
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Returns
                </a>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Categories</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Smartphones
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Appliances
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Electronics
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Home & Garden
                </a>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibent text-lg">Support</h4>
              <div className="space-y-2">
                <p className="text-gray-400">📞 1-800-SAVINGS</p>
                <p className="text-gray-400">✉️ support@savingvest.com</p>
                <p className="text-gray-400">🕒 24/7 Customer Service</p>
              </div>
            </div>
          </div>

          <div className="border-t border-gray-800 mt-12 pt-8 text-center">
            <p className="text-gray-400">&copy; 2024 SavingVest Mart. All rights reserved.</p>
          </div>
        </div>
      </footer>
    </div>
  )
}
