"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"
import { Slider } from "@/components/ui/slider"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { ShoppingCart, Search, User, Menu, Star, Filter, Grid, List, ChevronRight } from "lucide-react"

export default function ProductsPage() {
  const [viewMode, setViewMode] = useState<"grid" | "list">("grid")
  const [sortBy, setSortBy] = useState("featured")
  const [priceRange, setPriceRange] = useState([0, 2000])
  const [selectedCategories, setSelectedCategories] = useState<string[]>([])
  const [selectedBrands, setSelectedBrands] = useState<string[]>([])
  const [showFilters, setShowFilters] = useState(false)

  const allProducts = [
    {
      id: 1,
      name: "iPhone 15 Pro Max",
      price: 1199,
      originalPrice: 1299,
      image: "/iphone-15-pro-max.png",
      rating: 4.9,
      reviews: 2847,
      badge: "Best Seller",
      category: "Smartphones",
      brand: "Apple",
      description: "Latest iPhone with titanium design and advanced camera system",
    },
    {
      id: 2,
      name: "Samsung Galaxy S24 Ultra",
      price: 1099,
      originalPrice: 1199,
      image: "/samsung-galaxy-s24-ultra.png",
      rating: 4.8,
      reviews: 1923,
      badge: "New",
      category: "Smartphones",
      brand: "Samsung",
      description: "Flagship Android phone with S Pen and AI features",
    },
    {
      id: 3,
      name: "Dyson V15 Detect",
      price: 649,
      originalPrice: 749,
      image: "/dyson-v15-detect.png",
      rating: 4.7,
      reviews: 1456,
      badge: "Sale",
      category: "Home Appliances",
      brand: "Dyson",
      description: "Powerful cordless vacuum with laser dust detection",
    },
    {
      id: 4,
      name: "KitchenAid Stand Mixer",
      price: 379,
      originalPrice: 449,
      image: "/kitchenaid-mixer.png",
      rating: 4.9,
      reviews: 3241,
      badge: "Popular",
      category: "Kitchen Appliances",
      brand: "KitchenAid",
      description: "Professional-grade stand mixer for all your baking needs",
    },
    {
      id: 5,
      name: "MacBook Air M3",
      price: 1099,
      originalPrice: 1199,
      image: "/placeholder-1sqhv.png",
      rating: 4.8,
      reviews: 1876,
      badge: "New",
      category: "Electronics",
      brand: "Apple",
      description: "Ultra-thin laptop with M3 chip and all-day battery life",
    },
    {
      id: 6,
      name: "Sony WH-1000XM5",
      price: 349,
      originalPrice: 399,
      image: "/placeholder-ab7js.png",
      rating: 4.6,
      reviews: 2156,
      badge: "Sale",
      category: "Electronics",
      brand: "Sony",
      description: "Premium noise-canceling wireless headphones",
    },
    {
      id: 7,
      name: "Instant Pot Duo 7-in-1",
      price: 89,
      originalPrice: 119,
      image: "/placeholder-wzd47.png",
      rating: 4.5,
      reviews: 4521,
      badge: "Best Value",
      category: "Kitchen Appliances",
      brand: "Instant Pot",
      description: "Multi-functional electric pressure cooker",
    },
    {
      id: 8,
      name: "Google Pixel 8 Pro",
      price: 899,
      originalPrice: 999,
      image: "/google-pixel-smartphone.png",
      rating: 4.7,
      reviews: 1234,
      badge: "AI Powered",
      category: "Smartphones",
      brand: "Google",
      description: "Advanced AI photography and pure Android experience",
    },
    {
      id: 9,
      name: "Ninja Foodi Air Fryer",
      price: 199,
      originalPrice: 249,
      image: "/ninja-air-fryer.png",
      rating: 4.4,
      reviews: 2876,
      badge: "Sale",
      category: "Kitchen Appliances",
      brand: "Ninja",
      description: "Large capacity air fryer with multiple cooking functions",
    },
    {
      id: 10,
      name: 'iPad Pro 12.9"',
      price: 1099,
      originalPrice: 1199,
      image: "/ipad-pro-tablet.png",
      rating: 4.8,
      reviews: 1567,
      badge: "Pro",
      category: "Electronics",
      brand: "Apple",
      description: "Professional tablet with M2 chip and Liquid Retina display",
    },
    {
      id: 11,
      name: "Roomba j7+",
      price: 599,
      originalPrice: 699,
      image: "/irobot-roomba-vacuum.png",
      rating: 4.3,
      reviews: 1890,
      badge: "Smart",
      category: "Home Appliances",
      brand: "iRobot",
      description: "Smart robot vacuum with automatic dirt disposal",
    },
    {
      id: 12,
      name: 'LG OLED C3 55"',
      price: 1299,
      originalPrice: 1499,
      image: "/lg-oled-tv.png",
      rating: 4.9,
      reviews: 987,
      badge: "Premium",
      category: "Electronics",
      brand: "LG",
      description: "4K OLED smart TV with perfect blacks and vibrant colors",
    },
  ]

  const categories = ["Smartphones", "Electronics", "Kitchen Appliances", "Home Appliances"]
  const brands = ["Apple", "Samsung", "Dyson", "KitchenAid", "Sony", "Instant Pot", "Google", "Ninja", "iRobot", "LG"]

  const filteredProducts = allProducts.filter((product) => {
    const categoryMatch = selectedCategories.length === 0 || selectedCategories.includes(product.category)
    const brandMatch = selectedBrands.length === 0 || selectedBrands.includes(product.brand)
    const priceMatch = product.price >= priceRange[0] && product.price <= priceRange[1]
    return categoryMatch && brandMatch && priceMatch
  })

  const sortedProducts = [...filteredProducts].sort((a, b) => {
    switch (sortBy) {
      case "price-low":
        return a.price - b.price
      case "price-high":
        return b.price - a.price
      case "rating":
        return b.rating - a.rating
      case "newest":
        return b.id - a.id
      default:
        return 0
    }
  })

  const toggleCategory = (category: string) => {
    setSelectedCategories((prev) =>
      prev.includes(category) ? prev.filter((c) => c !== category) : [...prev, category],
    )
  }

  const toggleBrand = (brand: string) => {
    setSelectedBrands((prev) => (prev.includes(brand) ? prev.filter((b) => b !== brand) : [...prev, brand]))
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Navigation Header */}
      <header className="border-b bg-white/95 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-8">
              <h1 className="text-2xl font-bold text-emerald-600">SavingVest Mart</h1>
              <nav className="hidden md:flex items-center gap-6">
                <a href="/" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Home
                </a>
                <a href="/products" className="text-sm font-medium text-emerald-600">
                  Products
                </a>
                <a href="#" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Categories
                </a>
                <a href="#" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Deals
                </a>
              </nav>
            </div>

            <div className="flex items-center gap-4">
              <div className="hidden md:flex items-center gap-2 bg-gray-100 rounded-full px-4 py-2 w-80">
                <Search className="h-4 w-4 text-gray-500" />
                <input
                  type="text"
                  placeholder="Search products..."
                  className="bg-transparent border-none outline-none flex-1 text-sm"
                />
              </div>
              <Button variant="ghost" size="icon">
                <User className="h-5 w-5" />
              </Button>
              <Button variant="ghost" size="icon" className="relative">
                <ShoppingCart className="h-5 w-5" />
                <Badge className="absolute -top-2 -right-2 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs">
                  3
                </Badge>
              </Button>
              <Button variant="ghost" size="icon" className="md:hidden">
                <Menu className="h-5 w-5" />
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Breadcrumb */}
      <div className="bg-gray-50 py-4">
        <div className="container mx-auto px-4">
          <div className="flex items-center gap-2 text-sm text-gray-600">
            <a href="/" className="hover:text-emerald-600">
              Home
            </a>
            <ChevronRight className="h-4 w-4" />
            <span className="text-gray-900">Products</span>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-4 py-8">
        <div className="flex gap-8">
          {/* Filters Sidebar */}
          <div className={`${showFilters ? "block" : "hidden"} lg:block w-full lg:w-80 space-y-6`}>
            <div className="bg-white rounded-lg p-6 shadow-sm border">
              <h3 className="font-semibold text-lg mb-4">Filters</h3>

              {/* Price Range */}
              <div className="space-y-4 mb-6">
                <h4 className="font-medium">Price Range</h4>
                <div className="px-2">
                  <Slider
                    value={priceRange}
                    onValueChange={setPriceRange}
                    max={2000}
                    min={0}
                    step={50}
                    className="w-full"
                  />
                  <div className="flex justify-between text-sm text-gray-600 mt-2">
                    <span>${priceRange[0]}</span>
                    <span>${priceRange[1]}</span>
                  </div>
                </div>
              </div>

              {/* Categories */}
              <div className="space-y-4 mb-6">
                <h4 className="font-medium">Categories</h4>
                <div className="space-y-2">
                  {categories.map((category) => (
                    <div key={category} className="flex items-center space-x-2">
                      <Checkbox
                        id={category}
                        checked={selectedCategories.includes(category)}
                        onCheckedChange={() => toggleCategory(category)}
                      />
                      <label htmlFor={category} className="text-sm cursor-pointer">
                        {category}
                      </label>
                    </div>
                  ))}
                </div>
              </div>

              {/* Brands */}
              <div className="space-y-4">
                <h4 className="font-medium">Brands</h4>
                <div className="space-y-2 max-h-48 overflow-y-auto">
                  {brands.map((brand) => (
                    <div key={brand} className="flex items-center space-x-2">
                      <Checkbox
                        id={brand}
                        checked={selectedBrands.includes(brand)}
                        onCheckedChange={() => toggleBrand(brand)}
                      />
                      <label htmlFor={brand} className="text-sm cursor-pointer">
                        {brand}
                      </label>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>

          {/* Products Section */}
          <div className="flex-1">
            {/* Toolbar */}
            <div className="flex items-center justify-between mb-6 bg-white rounded-lg p-4 shadow-sm border">
              <div className="flex items-center gap-4">
                <Button variant="ghost" size="sm" onClick={() => setShowFilters(!showFilters)} className="lg:hidden">
                  <Filter className="h-4 w-4 mr-2" />
                  Filters
                </Button>
                <span className="text-sm text-gray-600">{sortedProducts.length} products found</span>
              </div>

              <div className="flex items-center gap-4">
                <Select value={sortBy} onValueChange={setSortBy}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Sort by" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="featured">Featured</SelectItem>
                    <SelectItem value="newest">Newest</SelectItem>
                    <SelectItem value="price-low">Price: Low to High</SelectItem>
                    <SelectItem value="price-high">Price: High to Low</SelectItem>
                    <SelectItem value="rating">Highest Rated</SelectItem>
                  </SelectContent>
                </Select>

                <div className="flex items-center gap-2">
                  <Button
                    variant={viewMode === "grid" ? "default" : "ghost"}
                    size="sm"
                    onClick={() => setViewMode("grid")}
                  >
                    <Grid className="h-4 w-4" />
                  </Button>
                  <Button
                    variant={viewMode === "list" ? "default" : "ghost"}
                    size="sm"
                    onClick={() => setViewMode("list")}
                  >
                    <List className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </div>

            {/* Products Grid/List */}
            <div className={viewMode === "grid" ? "grid md:grid-cols-2 xl:grid-cols-3 gap-6" : "space-y-4"}>
              {sortedProducts.map((product) => (
                <Card
                  key={product.id}
                  className={`group hover:shadow-xl transition-all duration-300 border-0 shadow-md ${
                    viewMode === "list" ? "flex flex-row" : ""
                  }`}
                >
                  <CardContent className="p-0">
                    {viewMode === "grid" ? (
                      <>
                        <div className="relative overflow-hidden rounded-t-lg">
                          <img
                            src={product.image || "/placeholder.svg"}
                            alt={product.name}
                            className="w-full h-64 object-cover group-hover:scale-105 transition-transform duration-300"
                          />
                          <Badge className="absolute top-3 left-3 bg-emerald-600 hover:bg-emerald-600">
                            {product.badge}
                          </Badge>
                        </div>

                        <div className="p-6 space-y-4">
                          <div>
                            <h3 className="font-semibold text-lg text-gray-900 mb-2">{product.name}</h3>
                            <p className="text-sm text-gray-600 mb-2">{product.description}</p>
                            <div className="flex items-center gap-2 mb-2">
                              <div className="flex items-center gap-1">
                                {[...Array(5)].map((_, i) => (
                                  <Star key={i} className="h-4 w-4 text-yellow-400 fill-current" />
                                ))}
                              </div>
                              <span className="text-sm text-gray-600">({product.reviews})</span>
                            </div>
                          </div>

                          <div className="flex items-center gap-2">
                            <span className="text-2xl font-bold text-gray-900">${product.price}</span>
                            {product.originalPrice > product.price && (
                              <span className="text-lg text-gray-500 line-through">${product.originalPrice}</span>
                            )}
                          </div>

                          <Button className="w-full bg-emerald-600 hover:bg-emerald-700">Add to Cart</Button>
                        </div>
                      </>
                    ) : (
                      <div className="flex">
                        <div className="relative w-48 h-48 overflow-hidden rounded-l-lg flex-shrink-0">
                          <img
                            src={product.image || "/placeholder.svg"}
                            alt={product.name}
                            className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                          />
                          <Badge className="absolute top-3 left-3 bg-emerald-600 hover:bg-emerald-600">
                            {product.badge}
                          </Badge>
                        </div>

                        <div className="flex-1 p-6 flex flex-col justify-between">
                          <div>
                            <h3 className="font-semibold text-xl text-gray-900 mb-2">{product.name}</h3>
                            <p className="text-gray-600 mb-3">{product.description}</p>
                            <div className="flex items-center gap-2 mb-3">
                              <div className="flex items-center gap-1">
                                {[...Array(5)].map((_, i) => (
                                  <Star key={i} className="h-4 w-4 text-yellow-400 fill-current" />
                                ))}
                              </div>
                              <span className="text-sm text-gray-600">({product.reviews} reviews)</span>
                            </div>
                          </div>

                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2">
                              <span className="text-2xl font-bold text-gray-900">${product.price}</span>
                              {product.originalPrice > product.price && (
                                <span className="text-lg text-gray-500 line-through">${product.originalPrice}</span>
                              )}
                            </div>
                            <Button className="bg-emerald-600 hover:bg-emerald-700">Add to Cart</Button>
                          </div>
                        </div>
                      </div>
                    )}
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Pagination */}
            <div className="flex justify-center mt-12">
              <div className="flex items-center gap-2">
                <Button variant="outline" disabled>
                  Previous
                </Button>
                <Button variant="default" className="bg-emerald-600">
                  1
                </Button>
                <Button variant="outline">2</Button>
                <Button variant="outline">3</Button>
                <Button variant="outline">Next</Button>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-16 mt-16">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-4 gap-8">
            <div className="space-y-4">
              <h3 className="text-2xl font-bold text-emerald-400">SavingVest Mart</h3>
              <p className="text-gray-400">Your trusted partner for premium products at unbeatable prices.</p>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Quick Links</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  About Us
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Contact
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Shipping Info
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Returns
                </a>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Categories</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Smartphones
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Appliances
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Electronics
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Home & Garden
                </a>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Support</h4>
              <div className="space-y-2">
                <p className="text-gray-400">📞 1-800-SAVINGS</p>
                <p className="text-gray-400">✉️ support@savingvest.com</p>
                <p className="text-gray-400">🕒 24/7 Customer Service</p>
              </div>
            </div>
          </div>

          <div className="border-t border-gray-800 mt-12 pt-8 text-center">
            <p className="text-gray-400">&copy; 2024 SavingVest Mart. All rights reserved.</p>
          </div>
        </div>
      </footer>
    </div>
  )
}
