<?php
$page_title = "Shopping Cart";
$page_description = "Review your selected items and proceed to checkout.";

require_once 'config/database.php';
include 'includes/header.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?redirect=" . urlencode($_SERVER['REQUEST_URI']));
    exit;
}

$conn = getDBConnection();
$user_id = $_SESSION['user_id'];

// Get cart items with product details
$sql = "SELECT ci.*, p.name, p.price, p.sale_price, p.image_url, p.stock_quantity, p.brand
        FROM cart_items ci
        JOIN products p ON ci.product_id = p.id
        WHERE ci.user_id = ? AND p.is_active = 1
        ORDER BY ci.created_at DESC";

$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$cart_items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Calculate totals
$subtotal = 0;
$total_items = 0;
foreach ($cart_items as $item) {
    $item_price = $item['sale_price'] ?: $item['price'];
    $subtotal += $item_price * $item['quantity'];
    $total_items += $item['quantity'];
}

$shipping = $subtotal >= 79200 ? 0 : 7992; // Free shipping over ₦79,200, otherwise ₦7,992
$tax = $subtotal * 0.075; // 7.5% VAT (Nigeria standard)
$total = $subtotal + $shipping + $tax;

$conn->close();
?>

<!-- Cart Header -->
<section class="bg-gray-100 py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <h1 class="text-3xl font-bold text-gray-900">Shopping Cart</h1>
        <p class="text-gray-600 mt-2"><?php echo $total_items; ?> item<?php echo $total_items !== 1 ? 's' : ''; ?> in your cart</p>
    </div>
</section>

<!-- Cart Content -->
<section class="py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <?php if (empty($cart_items)): ?>
            <!-- Empty Cart -->
            <div class="text-center py-16">
                <svg class="w-24 h-24 text-gray-400 mx-auto mb-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4m0 0L7 13m0 0l-1.5 6M7 13l-1.5 6m0 0h9m-9 0V9a2 2 0 012-2h2m5 0V7a2 2 0 012-2h2m0 0V5a2 2 0 012-2h2"></path>
                </svg>
                <h2 class="text-2xl font-bold text-gray-900 mb-4">Your cart is empty</h2>
                <p class="text-gray-600 mb-8">Looks like you haven't added any items to your cart yet.</p>
                <a href="products.php" class="bg-primary-500 text-white px-8 py-3 rounded-lg font-semibold hover:bg-primary-600">
                    Continue Shopping
                </a>
            </div>
        <?php else: ?>
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Cart Items -->
                <div class="lg:col-span-2">
                    <div class="bg-white rounded-lg shadow-sm overflow-hidden">
                        <div class="px-6 py-4 border-b border-gray-200">
                            <h2 class="text-lg font-semibold text-gray-900">Cart Items</h2>
                        </div>
                        
                        <div class="divide-y divide-gray-200">
                            <?php foreach ($cart_items as $item): ?>
                            <?php 
                            $item_price = $item['sale_price'] ?: $item['price'];
                            $item_total = $item_price * $item['quantity'];
                            ?>
                            <div class="p-6 flex items-center space-x-4" id="cart-item-<?php echo $item['id']; ?>">
                                <!-- Product Image -->
                                <div class="flex-shrink-0">
                                    <img src="<?php echo htmlspecialchars($item['image_url']); ?>" 
                                         alt="<?php echo htmlspecialchars($item['name']); ?>"
                                         class="w-20 h-20 object-cover rounded-lg">
                                </div>
                                
                                <!-- Product Details -->
                                <div class="flex-1 min-w-0">
                                    <h3 class="text-lg font-semibold text-gray-900 mb-1">
                                        <a href="product.php?id=<?php echo $item['product_id']; ?>" class="hover:text-primary-600">
                                            <?php echo htmlspecialchars($item['name']); ?>
                                        </a>
                                    </h3>
                                    <p class="text-sm text-gray-600 mb-2"><?php echo htmlspecialchars($item['brand']); ?></p>
                                    
                                    <!-- Price -->
                                    <div class="flex items-center space-x-2">
                                        <span class="text-lg font-bold text-gray-900">₦<?php echo number_format($item_price, 2); ?></span>
                                        <?php if ($item['sale_price']): ?>
                                            <span class="text-sm text-gray-500 line-through">₦<?php echo number_format($item['price'], 2); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <!-- Quantity Controls -->
                                <div class="flex items-center space-x-3">
                                    <button onclick="updateQuantity(<?php echo $item['id']; ?>, <?php echo $item['quantity'] - 1; ?>)"
                                            class="w-8 h-8 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50"
                                            <?php echo $item['quantity'] <= 1 ? 'disabled' : ''; ?>>
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"></path>
                                        </svg>
                                    </button>
                                    
                                    <span class="w-12 text-center font-medium" id="quantity-<?php echo $item['id']; ?>">
                                        <?php echo $item['quantity']; ?>
                                    </span>
                                    
                                    <button onclick="updateQuantity(<?php echo $item['id']; ?>, <?php echo $item['quantity'] + 1; ?>)"
                                            class="w-8 h-8 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50"
                                            <?php echo $item['quantity'] >= $item['stock_quantity'] ? 'disabled' : ''; ?>>
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                                        </svg>
                                    </button>
                                </div>
                                
                                <!-- Item Total -->
                                <div class="text-right">
                                    <p class="text-lg font-bold text-gray-900" id="total-<?php echo $item['id']; ?>">
                                        ₦<?php echo number_format($item_total, 2); ?>
                                    </p>
                                    <button onclick="removeFromCart(<?php echo $item['id']; ?>)"
                                            class="text-red-600 hover:text-red-800 text-sm font-medium mt-1">
                                        Remove
                                    </button>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Continue Shopping -->
                    <div class="mt-6">
                        <a href="products.php" class="text-primary-600 hover:text-primary-800 font-medium">
                            ← Continue Shopping
                        </a>
                    </div>
                </div>
                
                <!-- Order Summary -->
                <div class="lg:col-span-1">
                    <div class="bg-white rounded-lg shadow-sm p-6 sticky top-24">
                        <h2 class="text-lg font-semibold text-gray-900 mb-4">Order Summary</h2>
                        
                        <div class="space-y-3 mb-6">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Subtotal (<?php echo $total_items; ?> items)</span>
                                <span class="font-medium" id="cart-subtotal">₦<?php echo number_format($subtotal, 2); ?></span>
                            </div>
                            
                            <div class="flex justify-between">
                                <span class="text-gray-600">Shipping</span>
                                <span class="font-medium" id="cart-shipping">
                                    <?php echo $shipping > 0 ? '₦' . number_format($shipping, 2) : 'FREE'; ?>
                                </span>
                            </div>
                            
                            <?php if ($subtotal < 79200 && $subtotal > 0): ?>
                            <div class="text-sm text-gray-600 bg-blue-50 p-3 rounded-lg">
                                Add ₦<?php echo number_format(79200 - $subtotal, 2); ?> more for free shipping!
                            </div>
                            <?php endif; ?>
                            
                            <div class="flex justify-between">
                                <span class="text-gray-600">VAT (7.5%)</span>
                                <span class="font-medium" id="cart-tax">₦<?php echo number_format($tax, 2); ?></span>
                            </div>
                            
                            <div class="border-t border-gray-200 pt-3">
                                <div class="flex justify-between">
                                    <span class="text-lg font-semibold text-gray-900">Total</span>
                                    <span class="text-lg font-bold text-primary-600" id="cart-total">₦<?php echo number_format($total, 2); ?></span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Checkout Button -->
                        <a href="checkout.php" class="w-full bg-primary-500 text-white py-3 px-6 rounded-lg font-semibold hover:bg-primary-600 text-center block">
                            Proceed to Checkout
                        </a>
                        
                        <!-- Security Badge -->
                        <div class="mt-4 flex items-center justify-center text-sm text-gray-600">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path>
                            </svg>
                            Secure checkout
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
// Cart management functions
function updateQuantity(cartItemId, newQuantity) {
    if (newQuantity < 1) return;
    
    fetch('api/update_cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            cart_item_id: cartItemId,
            quantity: newQuantity
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update quantity display
            document.getElementById(`quantity-${cartItemId}`).textContent = newQuantity;
            
            // Update item total
            document.getElementById(`total-${cartItemId}`).textContent = `₦${data.item_total}`;
            
            // Update cart totals
            updateCartTotals(data.cart_totals);
            updateCartCount();
        } else {
            showNotification(data.message || 'Failed to update quantity', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('An error occurred. Please try again.', 'error');
    });
}

function removeFromCart(cartItemId) {
    if (!confirm('Are you sure you want to remove this item from your cart?')) {
        return;
    }
    
    fetch('api/remove_from_cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            cart_item_id: cartItemId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Remove item from DOM
            document.getElementById(`cart-item-${cartItemId}`).remove();
            
            // Update cart totals
            updateCartTotals(data.cart_totals);
            updateCartCount();
            
            // Reload page if cart is empty
            if (data.cart_totals.total_items === 0) {
                location.reload();
            }
            
            showNotification('Item removed from cart', 'success');
        } else {
            showNotification(data.message || 'Failed to remove item', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('An error occurred. Please try again.', 'error');
    });
}

function updateCartTotals(totals) {
    document.getElementById('cart-subtotal').textContent = `₦${totals.subtotal}`;
    document.getElementById('cart-shipping').textContent = totals.shipping > 0 ? `₦${totals.shipping}` : 'FREE';
    document.getElementById('cart-tax').textContent = `₦${totals.tax}`;
    document.getElementById('cart-total').textContent = `₦${totals.total}`;
}
</script>

<?php include 'includes/footer.php'; ?>
