<?php
$page_title = "My Wishlist";
$page_description = "View and manage your favorite products.";

require_once 'config/database.php';
include 'includes/header.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?redirect=" . urlencode($_SERVER['REQUEST_URI']));
    exit;
}

$conn = getDBConnection();
$user_id = $_SESSION['user_id'];

// Get wishlist items
$wishlist_sql = "SELECT p.*, w.added_at 
                 FROM wishlist w 
                 JOIN products p ON w.product_id = p.id 
                 WHERE w.user_id = ? 
                 ORDER BY w.added_at DESC";
$wishlist_stmt = $conn->prepare($wishlist_sql);
$wishlist_stmt->bind_param("i", $user_id);
$wishlist_stmt->execute();
$wishlist_items = $wishlist_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$conn->close();
?>

<section class="py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <!-- Header -->
        <div class="mb-8">
            <nav class="flex mb-4" aria-label="Breadcrumb">
                <ol class="flex items-center space-x-2">
                    <li><a href="index.php" class="text-gray-500 hover:text-gray-700">Home</a></li>
                    <li><span class="text-gray-400">/</span></li>
                    <li><a href="dashboard.php" class="text-gray-500 hover:text-gray-700">Dashboard</a></li>
                    <li><span class="text-gray-400">/</span></li>
                    <li><span class="text-gray-900 font-medium">Wishlist</span></li>
                </ol>
            </nav>
            <h1 class="text-3xl font-bold text-gray-900">My Wishlist</h1>
            <p class="text-gray-600 mt-2"><?php echo count($wishlist_items); ?> item<?php echo count($wishlist_items) !== 1 ? 's' : ''; ?> saved</p>
        </div>

        <?php if (empty($wishlist_items)): ?>
            <!-- Empty State -->
            <div class="text-center py-12">
                <svg class="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"></path>
                </svg>
                <h3 class="text-xl font-medium text-gray-900 mb-2">Your wishlist is empty</h3>
                <p class="text-gray-600 mb-6">Save items you love to your wishlist and shop them later.</p>
                <a href="products.php" class="bg-primary-500 text-white px-6 py-3 rounded-lg font-medium hover:bg-primary-600">
                    Browse Products
                </a>
            </div>
        <?php else: ?>
            <!-- Wishlist Items -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                <?php foreach ($wishlist_items as $item): ?>
                <div class="bg-white rounded-lg shadow-sm overflow-hidden group">
                    <div class="aspect-square bg-gray-100 relative">
                        <img src="<?php echo htmlspecialchars($item['image_url']); ?>" 
                             alt="<?php echo htmlspecialchars($item['name']); ?>"
                             class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-200">
                        <button onclick="removeFromWishlist(<?php echo $item['id']; ?>)" 
                                class="absolute top-3 right-3 p-2 bg-white rounded-full shadow-md hover:bg-red-50 transition-colors">
                            <svg class="w-5 h-5 text-red-500" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/>
                            </svg>
                        </button>
                    </div>
                    <div class="p-4">
                        <h3 class="text-lg font-semibold text-gray-900 mb-2 line-clamp-2">
                            <a href="product.php?id=<?php echo $item['id']; ?>" class="hover:text-primary-600">
                                <?php echo htmlspecialchars($item['name']); ?>
                            </a>
                        </h3>
                        <p class="text-gray-600 text-sm mb-3 line-clamp-2"><?php echo htmlspecialchars($item['description']); ?></p>
                        <div class="flex items-center justify-between mb-3">
                            <span class="text-2xl font-bold text-primary-600">$<?php echo number_format($item['price'], 2); ?></span>
                            <?php if ($item['stock_quantity'] > 0): ?>
                                <span class="text-sm text-green-600 font-medium">In Stock</span>
                            <?php else: ?>
                                <span class="text-sm text-red-600 font-medium">Out of Stock</span>
                            <?php endif; ?>
                        </div>
                        <div class="flex space-x-2">
                            <button onclick="addToCart(<?php echo $item['id']; ?>)" 
                                    class="flex-1 bg-primary-500 text-white py-2 px-4 rounded-lg font-medium hover:bg-primary-600 transition-colors <?php echo $item['stock_quantity'] <= 0 ? 'opacity-50 cursor-not-allowed' : ''; ?>"
                                    <?php echo $item['stock_quantity'] <= 0 ? 'disabled' : ''; ?>>
                                Add to Cart
                            </button>
                            <a href="product.php?id=<?php echo $item['id']; ?>" 
                               class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                                <svg class="w-5 h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                                </svg>
                            </a>
                        </div>
                        <p class="text-xs text-gray-500 mt-2">Added <?php echo date('M j, Y', strtotime($item['added_at'])); ?></p>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
function removeFromWishlist(productId) {
    if (confirm('Remove this item from your wishlist?')) {
        fetch('api/remove_from_wishlist.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ product_id: productId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Error removing item from wishlist');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error removing item from wishlist');
        });
    }
}

function addToCart(productId) {
    fetch('api/add_to_cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ 
            product_id: productId,
            quantity: 1
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update cart count in header
            updateCartCount();
            // Show success message
            showNotification('Item added to cart!', 'success');
        } else {
            alert(data.message || 'Error adding item to cart');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error adding item to cart');
    });
}
</script>

<?php include 'includes/footer.php'; ?>
